using System;
using System.Data;
using gov.va.med.vbecs;
using gov.va.med.vbecs.DAL;
using Method = System.Reflection.MethodBase;
using gov.va.med.vbecs.Common;
using System.Text.RegularExpressions;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

namespace gov.va.med.vbecs.BOL 
{

		#region Header

		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///<Developer>Carrie Van Stedum</Developer>
		///<Developer>Greg Lohse</Developer>
		///<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate> 8/20/2002</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		/// <summary>
		/// The LocalSupplier class holds all information related to a person, facility or organization that is a source of blood
		/// </summary>

		#endregion

	[Serializable] 
	public class LocalSupplier : BaseBusinessObject, IRecordStatusCode
	{
		/// <summary>
		/// OnColumnChanged
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="eventArgs"></param>
		private void OnColumnChanged(object sender, DataColumnChangeEventArgs eventArgs)
		{
			this.IsDirty = true;
		}


	#region Variables

		/// <summary>
		/// _objAddress
		/// </summary>
		private Address _objAddress;
		/// <summary>
		/// _dtLocalSupplier
		/// </summary>
		private System.Data.DataTable _dtLocalSupplier;
		/// <summary>
		/// _drFacility
		/// </summary>
		private System.Data.DataRow _drFacility;
		/// <summary>
		/// _hasAddress
		/// </summary>
		private bool _hasAddress = false;

	#endregion

	#region Constructors


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/6/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2505"> 
		///		<ExpectedInput>Guid, bool</ExpectedInput>
		///		<ExpectedOutput>LocalSupplier object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2509"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Local Supplier Contstructor
		/// </summary>
		/// <param name="localSupplierGuid"></param>
		/// <param name="activeOnly">Include Inactive facilities when searching database to build object</param>
		public LocalSupplier(System.Guid localSupplierGuid, bool activeOnly) 
		{
			_dtLocalSupplier = DAL.LocalSupplier.GetLocalSupplierDetailByGuid(localSupplierGuid, activeOnly);

			this.IsNew = (_dtLocalSupplier.Rows.Count == 0);
			
			if(this.IsNew)
			{
				this.MakeNew();
				if (localSupplierGuid != System.Guid.Empty)
				{
					this.LocalSupplierGuid = localSupplierGuid;
				}
			}
			else
			{
				this._drFacility = this._dtLocalSupplier.Rows[0];
				this.LoadFromDataRow(_dtLocalSupplier.Rows[0]);
				//Because of the new Local/National Supplier Lookup a National Row can be returned
				//when looking up Facilty by FDA#.  So manually check the code to see if it's a National
				//Facility being returned.. If so, then a new local supplier row is required...
				if (this.RecordStatusCode == Common.RecordStatusCode.NationalStandard)
				{
					this.Address.AddressGuid = System.Guid.NewGuid();
					this.Address.IsNew = true;
					this.IsNew = true;
					this.IsDirty = true;
				}
			}

			//This will handle our dirty flag setting
			this._drFacility.AcceptChanges();
			this._dtLocalSupplier.ColumnChanged += new DataColumnChangeEventHandler(this.OnColumnChanged);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/6/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2547"> 
		///		<ExpectedInput>string, bool</ExpectedInput>
		///		<ExpectedOutput>LocalSupplier object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2548"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Local Supplier Constructor
		/// </summary>
		/// <param name="registrationNumber"></param>
		/// <param name="activeOnly">Include Inactive facilities when searching database to build object</param>
		public LocalSupplier(string registrationNumber, bool activeOnly)
		{
			_dtLocalSupplier = DAL.LocalSupplier.GetLocalSupplierDetailByFdaRegNum(registrationNumber, activeOnly);
			
			this.IsNew = (_dtLocalSupplier.Rows.Count == 0);
			
			if(this.IsNew)
			{
				this.MakeNew();
				this.FDARegistrationNumber = registrationNumber;
			}
			else
			{
				this._drFacility = this._dtLocalSupplier.Rows[0];
				this.LoadFromDataRow(_dtLocalSupplier.Rows[0]);
				//Because of the new Local/National Supplier Lookup a National Row can be returned
				//when looking up Facilty by FDA#.  So manually check the code to see if it's a National
				//Facility being returned.. If so, then a new local supplier row is required...
				if (this.RecordStatusCode == Common.RecordStatusCode.NationalStandard)
				{
					this.Address.AddressGuid = System.Guid.NewGuid();
					this.Address.IsNew = true;
					this.IsNew = true;
					this.IsDirty = true;
				}
			}

			//This will handle our dirty flag setting
			this._drFacility.AcceptChanges();
			this._dtLocalSupplier.ColumnChanged += new DataColumnChangeEventHandler(this.OnColumnChanged);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7436"> 
		///		<ExpectedInput>bool, string</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7437"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Allows Facilty to be created with only ICCBBA number
		/// </summary>
		/// <param name="iccbaRegNumber"></param>
		/// <param name="activeOnly"></param>
		public LocalSupplier(bool activeOnly, string iccbaRegNumber)
		{
			_dtLocalSupplier = DAL.LocalSupplier.GetLocalSupplierDetailByIccbaRegNum(iccbaRegNumber, activeOnly);
			
			this.IsNew = (_dtLocalSupplier.Rows.Count == 0);
			
			if(this.IsNew)
			{
				this.MakeNew();
				this.IccbbaRegistrationNumber = iccbaRegNumber;
			}
			else
			{
				this._drFacility = this._dtLocalSupplier.Rows[0];
				this.LoadFromDataRow(_dtLocalSupplier.Rows[0]);
				//Because of the new Local/National Supplier Lookup a National Row can be returned
				//when looking up Facilty by FDA#.  So manually check the code to see if it's a National
				//Facility being returned.. If so, then a new local supplier row is required...
				if (this.RecordStatusCode == Common.RecordStatusCode.NationalStandard)
				{
					this.Address.AddressGuid = System.Guid.NewGuid();
					this.Address.IsNew = true;
					this.IsNew = true;
					this.IsDirty = true;
				}
			}

			//This will handle our dirty flag setting
			this._drFacility.AcceptChanges();
			this._dtLocalSupplier.ColumnChanged += new DataColumnChangeEventHandler(this.OnColumnChanged);
		}
	#endregion

	#region Methods


		/// <summary>
		/// Logic that will populate the class 
		/// </summary>
		private void MakeNew()
		{
			this._drFacility = this._dtLocalSupplier.NewRow();

			this.Address = new Address();

			this._drFacility[TABLE.LocalSupplier.LocalSupplierGuid] = System.Guid.NewGuid();
			this._drFacility[TABLE.LocalSupplier.FacilityName] = string.Empty;
			this._drFacility[TABLE.LocalSupplier.FdaRegistrationNumber] = string.Empty;
			this._drFacility[TABLE.LocalSupplier.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			this._drFacility[TABLE.LocalSupplier.RowVersion] = System.DBNull.Value;
			this._drFacility[Common.DatabaseConstants.ArtificialColumnNames.AddressRowVersion] = System.DBNull.Value;
			//We used to default this to LocalSupplier -- but that meant it was local and active... now we're making them do it.
			this._drFacility[TABLE.LocalSupplier.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive);
			this._dtLocalSupplier.Rows.Add(this._drFacility);

			this.IsCollectionFacility = true;
			this.IsDirty = true;
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/6/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2549"> 
		///		<ExpectedInput>string , string</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2550"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Retrieves a datatable of local suppliers based on input parameters
		/// </summary>
		/// <param name="facilityName"></param>
		/// <param name="fdaRegistrationNumber"></param>
		/// <returns>DataTable</returns>
		public static DataTable GetLocalSupplierLookupList(string facilityName, string fdaRegistrationNumber)
		{
			return DAL.LocalSupplier.GetLocalSupplierLookupList(facilityName, fdaRegistrationNumber);
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5026"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Non-empty DataTable containing list of suppliers.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5027"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves a list of all national facilities.
		/// </summary>
		/// <returns><see cref="DataTable"/> containing list of all national facilities.</returns>		
		public static DataTable GetFacilityNationalLookupList()
		{
			return DAL.LocalSupplier.GetFacilityNationalLookupList();
		}


		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2551"> 
		///		<ExpectedInput>string, bool</ExpectedInput>
		///		<ExpectedOutput>Datatable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2554"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets the lcoal supplier information for a given FDAReg#
		/// </summary>
		/// <param name="fdaRegistrationNumber"></param>
		/// <param name="activeOnly">Include inactive locl suppliers when quereying data</param>
		/// <returns>DataTable</returns>
		public DataTable GetLocalSupplierDetailByFdaRegNum(string fdaRegistrationNumber, bool activeOnly)
		{
			_dtLocalSupplier = DAL.LocalSupplier.GetLocalSupplierDetailByFdaRegNum(fdaRegistrationNumber, activeOnly);

			if(_dtLocalSupplier.Rows.Count > 0)
			{
				this.RecordStatusCode = Common.Utility.GetRecordStatusCodeFromString(_dtLocalSupplier.Rows[0][TABLE.LocalSupplier.RecordStatusCode].ToString());
			}
			return _dtLocalSupplier;
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2555"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Datatable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2567"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets a datatable of available local suppliers for the current user's division
		/// </summary>
		/// <returns>DataTable</returns>
		public static DataTable GetLocalSuppliersForDivision()
		{
			return DAL.LocalSupplier.GetLocalSuppliersForDivision();
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2568"> 
		///		<ExpectedInput>valid fdaregistration number</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2571"> 
		///		<ExpectedInput>invalid fdaregistration number</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns true if the FDA Reg# exists on the database
		/// </summary>
		/// <param name="fdaRegistrationNumber"></param>
		/// <returns>bool</returns>
		public static bool IsValidFdaRegistrationNumber(string fdaRegistrationNumber)
		{
			return DAL.LocalSupplier.IsValidFdaRegistrationNumber(fdaRegistrationNumber);
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2572"> 
		///		<ExpectedInput>UpdateFunction</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2579"> 
		///		<ExpectedInput>UpdateFunction</ExpectedInput>
		///		<ExpectedOutput>SqlException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Inserts a new local supplier.  Returns true is success
		/// </summary>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <returns>bool</returns>
		public bool InsertLocalSupplier(Common.UpdateFunction lastUpdateFunctionId)
		{
			System.Data.DataTable dtInsertLS = DAL.LocalSupplier.GetEmptyLocalSupplierTableSchema(false);
			System.Data.DataRow drInsLS = this.LoadDataRowFromThis(dtInsertLS.NewRow());
			dtInsertLS.Rows.Add(drInsLS);

			dtInsertLS = Common.Utility.AppendLastUpdateInformation(dtInsertLS, lastUpdateFunctionId);

			System.Data.DataTable dtAddress = DAL.Address.GetEmptyAddressTableSchema(!this.Address.IsNew);
			System.Data.DataRow drAdd = this.Address.LoadDataRowFromThis(dtAddress.NewRow());
			dtAddress.Rows.Add(drAdd);

			dtAddress = Common.Utility.AppendLastUpdateInformation(dtAddress, lastUpdateFunctionId);

			return DAL.LocalSupplier.InsertLocalSupplier(dtInsertLS, dtAddress, this.Address.IsNew);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/6/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2580"> 
		///		<ExpectedInput>UpdateFunction</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2581"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Update the curent local supplier.  Returns true is success
		/// </summary>
		/// <returns>bool</returns>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		public bool UpdateLocalSupplier(Common.UpdateFunction lastUpdateFunctionId)
		{
			System.Data.DataTable dtUpdateLS = DAL.LocalSupplier.GetEmptyLocalSupplierTableSchema(true);
			System.Data.DataRow drUpdLS = this.LoadDataRowFromThis(dtUpdateLS.NewRow());
			dtUpdateLS.Rows.Add(drUpdLS);

			dtUpdateLS = Common.Utility.AppendLastUpdateInformation(dtUpdateLS, lastUpdateFunctionId);

			//An update to a Supplier will always = an update to an address -- Insert is done on Local Insert
			System.Data.DataTable dtUpdateAddress = DAL.Address.GetEmptyAddressTableSchema(true);
			System.Data.DataRow drUpdAdd = this.Address.LoadDataRowFromThis(dtUpdateAddress.NewRow());
			dtUpdateAddress.Rows.Add(drUpdAdd);

			dtUpdateAddress = Common.Utility.AppendLastUpdateInformation(dtUpdateAddress, lastUpdateFunctionId);

			return DAL.LocalSupplier.UpdateLocalSupplier(dtUpdateLS, dtUpdateAddress);
		}


		///<Developers>
		///<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2582"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>Populated DataRow</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="2586"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Load a datarow with object information
		/// </summary>
		/// <param name="dtRow">DataRow to load into, which then is returned</param>
		/// <returns>DataRow</returns>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.LocalSupplierGuid))
				dtRow[TABLE.LocalSupplier.LocalSupplierGuid] = this.LocalSupplierGuid;
			
			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.FacilityGuid))
			{
				if (this.FacilityGuid != System.Guid.Empty)
					dtRow[TABLE.LocalSupplier.FacilityGuid] = this.FacilityGuid;
			}

			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.FdaRegistrationNumber))
				dtRow[TABLE.LocalSupplier.FdaRegistrationNumber] = this.FDARegistrationNumber;
			
			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.FacilityName))
				dtRow[TABLE.LocalSupplier.FacilityName]= this.FacilityName;
			
			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.IccbbaRegistrationNumber))			
				dtRow[TABLE.LocalSupplier.IccbbaRegistrationNumber] = this.IccbbaRegistrationNumber;
	
			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.RowVersion))			
				dtRow[TABLE.LocalSupplier.RowVersion] = this.RowVersion;
			
			if (dtRow.Table.Columns.Contains(TABLE.Address.AddressLine1))			
				dtRow[TABLE.Address.AddressLine1]=this.Address.AddressLine1;
			
			if (dtRow.Table.Columns.Contains(TABLE.Address.AddressLine2))			
				dtRow[TABLE.Address.AddressLine2] = this.Address.AddressLine2;
			
			if (dtRow.Table.Columns.Contains(TABLE.Address.AddressLine3))			
				dtRow[TABLE.Address.AddressLine3] = this.Address.AddressLine3;
			
			if (dtRow.Table.Columns.Contains(TABLE.Address.AddressCity))			
				dtRow[TABLE.Address.AddressCity] = this.Address.City;
			
			if (dtRow.Table.Columns.Contains(TABLE.Address.StateCode))			
				dtRow[TABLE.Address.StateCode]  = this.Address.StateCode;
			
			if (dtRow.Table.Columns.Contains(TABLE.Address.AddressZipCode))			
				dtRow[TABLE.Address.AddressZipCode]= this.Address.ZipCode;
			
			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.PhoneNumber))			
				dtRow[TABLE.LocalSupplier.PhoneNumber] = this.PhoneNumber;
			
			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.FaxNumber))			
				dtRow[TABLE.LocalSupplier.FaxNumber] = this.FaxNumber;
			
			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.LabelUnitIdTrimIndicator))			
				dtRow[TABLE.LocalSupplier.LabelUnitIdTrimIndicator] = this.LabelUnitIDTrimIndicator;
			
			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.FacilityPrefix))			
				dtRow[TABLE.LocalSupplier.FacilityPrefix] = this.FacilityPrefix;
			
			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.RecordStatusCode))			
				dtRow[TABLE.LocalSupplier.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(this.RecordStatusCode);
			
			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.DivisionCode))			
				dtRow[TABLE.LocalSupplier.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;

			if (dtRow.Table.Columns.Contains(TABLE.Address.AddressGuid))			
				dtRow[TABLE.Address.AddressGuid] = this.Address.AddressGuid;

			if (dtRow.Table.Columns.Contains(Common.DatabaseConstants.ArtificialColumnNames.AddressRowVersion))			
				dtRow[Common.DatabaseConstants.ArtificialColumnNames.AddressRowVersion] = this.Address.RowVersion;

			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.TestingIndicator))
				dtRow[TABLE.LocalSupplier.TestingIndicator] = this.IsTestingFacility;

			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.CollectionIndicator))
				dtRow[TABLE.LocalSupplier.CollectionIndicator] = this.IsCollectionFacility;

			
			//lastupdatefuncion
			return dtRow;
		}


		/// <summary>
		/// Loads the current object with information from datarow parameter
		/// </summary>
		/// <param name="dtRow">DataRow to load from</param>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.LocalSupplierGuid))
			{
				if(!dtRow.IsNull(TABLE.LocalSupplier.LocalSupplierGuid))
				{
					this.LocalSupplierGuid = (System.Guid)dtRow[TABLE.LocalSupplier.LocalSupplierGuid];
				}
				else
				{
					this.LocalSupplierGuid = System.Guid.NewGuid();
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.FacilityGuid))
			{
				if(!dtRow.IsNull(TABLE.LocalSupplier.FacilityGuid))
				{
					this.FacilityGuid = (System.Guid)dtRow[TABLE.LocalSupplier.FacilityGuid];
				}
				else
				{
					this._drFacility[TABLE.LocalSupplier.FacilityGuid] = DBNull.Value;
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.FacilityName))
			{
				if(!dtRow.IsNull(TABLE.LocalSupplier.FacilityName))
				{
					this.FacilityName = dtRow[TABLE.LocalSupplier.FacilityName].ToString();
				}
				else
				{
					this.FacilityName = String.Empty;
				}

			}

			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.FdaRegistrationNumber))
			{
				if(!dtRow.IsNull(TABLE.LocalSupplier.FdaRegistrationNumber))
				{
					this.FDARegistrationNumber = dtRow[TABLE.LocalSupplier.FdaRegistrationNumber].ToString();
				}
				else
				{
					this.FDARegistrationNumber = String.Empty;
				}

			}

			
			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.IccbbaRegistrationNumber))
			{
				if(!dtRow.IsNull(TABLE.LocalSupplier.IccbbaRegistrationNumber))
				{
					this.IccbbaRegistrationNumber = dtRow[TABLE.LocalSupplier.IccbbaRegistrationNumber].ToString();
				}
				else
				{
					this.IccbbaRegistrationNumber = String.Empty;
				}

			}

			this._hasAddress = false;
			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.AddressGuid))
			{
				if(!dtRow.IsNull(TABLE.LocalSupplier.AddressGuid))
				{
					this.Address = new BOL.Address(dtRow);
					this._hasAddress = true;
				}
				else
				{
					this.Address = new BOL.Address();
				}					
			}

			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.LabelUnitIdTrimIndicator))
			{
				if(!dtRow.IsNull(TABLE.LocalSupplier.LabelUnitIdTrimIndicator))
				{
					this.LabelUnitIDTrimIndicator = System.Convert.ToBoolean(dtRow[TABLE.LocalSupplier.LabelUnitIdTrimIndicator]);
				}
				else
				{
					this.LabelUnitIDTrimIndicator = false;
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.FacilityPrefix))
			{
				if(!dtRow.IsNull(TABLE.LocalSupplier.FacilityPrefix))
				{
					this.FacilityPrefix = dtRow[TABLE.LocalSupplier.FacilityPrefix].ToString();
				}
				else
				{
					this.FacilityPrefix = String.Empty;
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.PhoneNumber))
			{
				if(!dtRow.IsNull(TABLE.LocalSupplier.PhoneNumber))
				{
					this.PhoneNumber = dtRow[TABLE.LocalSupplier.PhoneNumber].ToString();
				}
				else
				{
					this.PhoneNumber = String.Empty;
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.FaxNumber))
			{
				if(!dtRow.IsNull(TABLE.LocalSupplier.FaxNumber))
				{
					this.FaxNumber = dtRow[TABLE.LocalSupplier.FaxNumber].ToString();
				}
				else
				{
					this.FaxNumber = String.Empty;
				}
			}

			//Doing this field different, cuz 1 indicator is required, and setting the property
			//directly will cause the check to fail because _drFacility["CollectionIndicator'] will be null
			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.TestingIndicator))
			{
				if(!dtRow.IsNull(TABLE.LocalSupplier.TestingIndicator))
				{
					this._drFacility[TABLE.LocalSupplier.TestingIndicator] = System.Convert.ToBoolean(dtRow[TABLE.LocalSupplier.TestingIndicator]);
				}
				else
				{
					this._drFacility[TABLE.LocalSupplier.TestingIndicator] = false;
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.CollectionIndicator))
			{
				if(!dtRow.IsNull(TABLE.LocalSupplier.CollectionIndicator))
				{
					this.IsCollectionFacility = System.Convert.ToBoolean(dtRow[TABLE.LocalSupplier.CollectionIndicator]);
				}
				else
				{
					this.IsCollectionFacility = true;
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.RecordStatusCode))
			{
				if(!dtRow.IsNull(TABLE.LocalSupplier.RecordStatusCode))
				{
					this.RecordStatusCode = Common.Utility.GetRecordStatusCodeFromString(dtRow[TABLE.LocalSupplier.RecordStatusCode].ToString());
				}
				else
				{
					this.RecordStatusCode = Common.RecordStatusCode.Unknown;
				}
			}

			if (dtRow.Table.Columns.Contains(TABLE.LocalSupplier.DivisionCode))
			{
				if(!dtRow.IsNull(TABLE.LocalSupplier.DivisionCode))
				{
					this._drFacility[TABLE.LocalSupplier.DivisionCode] = dtRow[TABLE.LocalSupplier.DivisionCode].ToString();
				}
				else
				{
					this._drFacility[TABLE.LocalSupplier.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
				}
			}
			if(dtRow.Table.Columns.Contains(TABLE.LocalSupplier.RowVersion))
			{
				if(!dtRow.IsNull(TABLE.LocalSupplier.RowVersion))
				{
					this.RowVersion= (byte[]) dtRow[TABLE.LocalSupplier.RowVersion];
				}
			}
		}	
		
		///<Developers>
		///<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2587"> 
		///		<ExpectedInput>Valid fdaregistration number</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="2588"> 
		///		<ExpectedInput>InValid fdaregistration number</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Checks the National/Local table for a FDARegistration#
		/// </summary>
		/// <param name="fdaRegistrationNumber"></param>
		/// <returns></returns>
		public static bool IsNewFDARegistrationNumber(string fdaRegistrationNumber)
		{
			DataTable dtTmp = DAL.LocalSupplier.GetLocalSupplierDetailByFdaRegNum(fdaRegistrationNumber, false);

			return (dtTmp.Rows.Count == 0);
		}


	#endregion

	#region Properties

		///<Developers>
		///<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2595"> 
		///		<ExpectedInput>guid</ExpectedInput>
		///		<ExpectedOutput>guid</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="2597"> 
		///		<ExpectedInput>Empty Guid</ExpectedInput>
		///		<ExpectedOutput>BrokenRule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// LocalSupplierGuid field from Local Supplier table
		/// </summary>
		public System.Guid LocalSupplierGuid
		{
			get
			{
				if (this._drFacility.IsNull(TABLE.LocalSupplier.LocalSupplierGuid))
				{
					this._drFacility[TABLE.LocalSupplier.LocalSupplierGuid] = System.Guid.NewGuid();
				}

				return (System.Guid)this._drFacility[TABLE.LocalSupplier.LocalSupplierGuid];
			}
			set
			{
				if (value == System.Guid.Empty)
				{
					RuleBroken(Method.GetCurrentMethod().Name, true);
					return;
				}

				this._drFacility[TABLE.LocalSupplier.LocalSupplierGuid] = value;
				RuleBroken(Method.GetCurrentMethod().Name, false);
			}
		}

		///<Developers>
		///<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2599"> 
		///		<ExpectedInput>guid</ExpectedInput>
		///		<ExpectedOutput>guid</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="2600"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// The FacilityGuid from the "national" Facility table.  Can be null for new local suppliers.
		/// </summary>
		public System.Guid FacilityGuid
		{
			get
			{
				if (this._drFacility.IsNull(TABLE.LocalSupplier.FacilityGuid))
				{
					return System.Guid.Empty;
				}
				return (System.Guid)this._drFacility[TABLE.LocalSupplier.FacilityGuid];
			}
			set
			{
				this._drFacility[TABLE.LocalSupplier.FacilityGuid] = value;
			}
		}

		///<Developers>
		///<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2601"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="2602"> 
		///		<ExpectedInput>Invalid String</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// FacilityName
		/// </summary>
		public string FacilityName
		{
			get
			{
				if (this._drFacility.IsNull(TABLE.LocalSupplier.FacilityName))
				{
					this._drFacility[TABLE.LocalSupplier.FacilityName] = String.Empty;
				}
				
				return this._drFacility[TABLE.LocalSupplier.FacilityName].ToString();
			}
			set
			{
				if (value.Length > 120)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
											Method.GetCurrentMethod().Name, true), "120").ResString;
					RuleBroken("FacilityIndicator", true);
					throw new BOL.BusinessObjectException(errorMessage);
				}
				
				if (value.Trim().Length == 0)
				{
					RuleBroken(Method.GetCurrentMethod().Name, true);
					return;
				}

				this._drFacility[TABLE.LocalSupplier.FacilityName] = value;	
				RuleBroken(Method.GetCurrentMethod().Name, false);
			}
		}

		///<Developers>
		///<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2610"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="2611"> 
		///		<ExpectedInput>Invalid String</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// FDARegistrationNumber
		/// BR_6.08 - Facility must be 7 or 10 chars
		/// </summary>
		public string FDARegistrationNumber
		{
			get
			{
				if (this._drFacility.IsNull(TABLE.LocalSupplier.FdaRegistrationNumber))
				{
					this._drFacility[TABLE.LocalSupplier.FdaRegistrationNumber] = String.Empty;
				}

				return this._drFacility[TABLE.LocalSupplier.FdaRegistrationNumber].ToString().Trim();

			}
			set
			{
				//BR_6.08 - Regular Expression only allows 7 or 10 chars
				Regex _regexFDAReg = Common.RegularExpressions.FDARegistrationNumber();

				if ((value.Trim().Length > 0) && !_regexFDAReg.IsMatch(value.Trim()))
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.InvalidPropertyOrMethod(
											Common.Utility.ParseStringwithSpaces(Method.GetCurrentMethod().Name, true),
											value).ResString;
					RuleBroken(Method.GetCurrentMethod().Name, true);
					throw new BOL.BusinessObjectException(errorMessage);

				}

				if (value.Trim().Length == 0)
				{
					RuleBroken(Method.GetCurrentMethod().Name, true);
					return;
				}

				this._drFacility[TABLE.LocalSupplier.FdaRegistrationNumber] = value;

				RuleBroken(Method.GetCurrentMethod().Name, false);
			}
		}


		///<Developers>
		///<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2612"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="2613"> 
		///		<ExpectedInput>Invalid String</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// FacilityPrefix - 2character prefix that indicates the facility
		/// </summary>
		public string FacilityPrefix
		{
			get
			{
				if (this._drFacility.IsNull(TABLE.LocalSupplier.FacilityPrefix))
				{
					this._drFacility[TABLE.LocalSupplier.FacilityPrefix] = String.Empty;
				}

				return this._drFacility[TABLE.LocalSupplier.FacilityPrefix].ToString().Trim();
			}
			set
			{

				switch (value.Trim().Length)
				{
						//hopefully length of 0 = emptystring
					case 0:
						break;

						//1 character is not good enough my friend
					case 1:
						RuleBroken(Method.GetCurrentMethod().Name, true);
						return;

						//Check if the entry is numeric
					case 2:
						Regex labelUnitIdEx = Common.RegularExpressions.EyeReadablePrefixNumeric();
						
//						if (this.LabelUnitIDTrimIndicator)
//							labelUnitIdEx = Common.RegularExpressions.EyeReadablePrefixAlphaNumeric();
//						else
//							labelUnitIdEx = Common.RegularExpressions.EyeReadablePrefixNumeric();

						if (!labelUnitIdEx.IsMatch(value))
						{
							RuleBroken(Method.GetCurrentMethod().Name, true);
							throw new BOL.BusinessObjectException("Prefix entered contains invalid characters.");
						}
						break;

						//Somehow the length is > 2
					default:
						RuleBroken(Method.GetCurrentMethod().Name, true);
						return;
				}

				this._drFacility[TABLE.LocalSupplier.FacilityPrefix] = value;
				RuleBroken(Method.GetCurrentMethod().Name, false);
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/30/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="285"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>ICCBBA registration number</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="292"> 
		///		<ExpectedInput>Invalid ICCBBA registration number</ExpectedInput>
		///		<ExpectedOutput>BusinessLayerException raised</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets ICCBBA Registration Number of a facility
		/// BR_6.11
		/// </summary>
		public string IccbbaRegistrationNumber
		{
			get
			{
				if (this._drFacility.IsNull(TABLE.LocalSupplier.IccbbaRegistrationNumber))
				{
					return(string.Empty);
				}

				return(this._drFacility[TABLE.LocalSupplier.IccbbaRegistrationNumber].ToString());
			}
			set
			{
				Regex iccbbaRegex = Common.RegularExpressions.IccbbaRegistrationNumber();
				
				//BR_6.11 - validate the ICCBBA format
				if ((value.Trim().Length > 0) && (!iccbbaRegex.IsMatch(value)))
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.InvalidPropertyOrMethod(
						Common.Utility.ParseStringwithSpaces(Method.GetCurrentMethod().Name, true ),
						value ).ResString;

					throw new BOL.BusinessObjectException(errorMessage);
				}
				
				if (this._drFacility.IsNull(TABLE.LocalSupplier.IccbbaRegistrationNumber) || (value != this._drFacility[TABLE.LocalSupplier.IccbbaRegistrationNumber].ToString()))
				{
					this._drFacility[TABLE.LocalSupplier.IccbbaRegistrationNumber] = value;
					this.IsDirty = true;
				}

				//TODO: Not required yet -- Needs SME talk - 12/30/2004
				//RuleBroken( MethodBase.GetCurrentMethod().Name, false);
			}
		}


		///<Developers>
		///<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2617"> 
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="2619"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// LabelUnitIDTrimIndicator
		/// </summary>
		public bool LabelUnitIDTrimIndicator
		{
			get 
			{
				if (this._drFacility.IsNull(TABLE.LocalSupplier.LabelUnitIdTrimIndicator))
				{
					this._drFacility[TABLE.LocalSupplier.LabelUnitIdTrimIndicator] = false;
				}

				return (bool)this._drFacility[TABLE.LocalSupplier.LabelUnitIdTrimIndicator];
			}
			set 
			{
				this._drFacility[TABLE.LocalSupplier.LabelUnitIdTrimIndicator] = value;
			}
		}

		///<Developers>
		///<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2625"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="2626"> 
		///		<ExpectedInput>INvalid string</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// PhoneNumber
		/// </summary>
		public string PhoneNumber
		{
			get 
			{
				if (this._drFacility.IsNull(TABLE.LocalSupplier.PhoneNumber))
				{
					this._drFacility[TABLE.LocalSupplier.PhoneNumber] = String.Empty;
				}

				return this._drFacility[TABLE.LocalSupplier.PhoneNumber].ToString().Trim();
			}
			set
			{
				if (value.Length > 14)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
											Method.GetCurrentMethod().Name, true), "14").ResString;
					throw new BOL.BusinessObjectException(errorMessage);
				}

				Regex phoneEx = Common.RegularExpressions.PhoneNumber();

				if (value.Trim().Length > 0 && !phoneEx.IsMatch(value))				
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.InvalidFormat(Common.Utility.ParseStringwithSpaces(
						Method.GetCurrentMethod().Name, true)).ResString;

					throw new BOL.BusinessObjectException(errorMessage);
				}

				this._drFacility[TABLE.LocalSupplier.PhoneNumber] = value;
			}
		}

		///<Developers>
		///<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2627"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="2629"> 
		///		<ExpectedInput>INvalid string</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// FaxNumber
		/// </summary>
		public string FaxNumber
		{
			get
			{
				if (this._drFacility.IsNull(TABLE.LocalSupplier.FaxNumber))
				{
					this._drFacility[TABLE.LocalSupplier.FaxNumber] = String.Empty;
				}

				return this._drFacility[TABLE.LocalSupplier.FaxNumber].ToString().Trim();
			}
			set
			{
				if (value.Length > 14)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
											Method.GetCurrentMethod().Name, true), "14").ResString;
					throw new BOL.BusinessObjectException(errorMessage);
				}

				Regex faxEx = Common.RegularExpressions.PhoneNumber();

				if (value.Trim().Length > 0 && !faxEx.IsMatch(value))
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.InvalidFormat(Common.Utility.ParseStringwithSpaces(
						Method.GetCurrentMethod().Name, true)).ResString;

					throw new BOL.BusinessObjectException(errorMessage);
				}

				this._drFacility[TABLE.LocalSupplier.FaxNumber] = value;
			}
		}

		///<Developers>
		///<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2631"> 
		///		<ExpectedInput>Address object</ExpectedInput>
		///		<ExpectedOutput>Address object</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="2635"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Address
		/// </summary>
		public Address Address
		{
			get
			{
				return _objAddress;
			}
			set
			{
				_objAddress = value;				
			}
		}

		///<Developers>
		///<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2637"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="2639"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Does this local supplier have a facility
		/// </summary>
		public bool HasAddress
		{
			get
			{
				return this._hasAddress;
			}
		}

		///<Developers>
		///<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2641"> 
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="2643"> 
		///		<ExpectedInput>false</ExpectedInput>
		///		<ExpectedOutput>Rule Broken</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// IS the local supplier a collection facility
		/// BR_6.09 - Facility must be Collection or Testing
		/// </summary>
		public bool IsCollectionFacility
		{
			get
			{
				if (this._drFacility.IsNull(TABLE.LocalSupplier.CollectionIndicator))
				{
					this._drFacility[TABLE.LocalSupplier.CollectionIndicator] = true;
				}

				return (bool)this._drFacility[TABLE.LocalSupplier.CollectionIndicator];
			}
			set
			{
				//Need to set the dr before validtaing since these to Indicator checks rely on each other
				//Needed for the case when a user checks one/then unchecks then checks another
				this._drFacility[TABLE.LocalSupplier.CollectionIndicator] = value;

				//BR_6.09 - Facility must be Collection or Testing
				if ((value == false) && (this.IsTestingFacility == false))
				{
//					string errorMessage =  Common.StrRes.SysErrMsg.Common.FacilityTypeIndicatorRequired().ResString;
					RuleBroken("FacilityIndicator", true);
					return;
//					throw new BOL.BusinessObjectException(errorMessage);
				}
				RuleBroken("FacilityIndicator", false);
			}
		}
		
		///<Developers>
		///<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2647"> 
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="2648"> 
		///		<ExpectedInput>false</ExpectedInput>
		///		<ExpectedOutput>Rule Broken</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Is the current local supplier a testing facility
		/// BR_6.09 - Facility must be Collection or Testing
		/// </summary>
		public bool IsTestingFacility
		{
			get
			{
				if (this._drFacility.IsNull(TABLE.LocalSupplier.TestingIndicator))
				{
					this._drFacility[TABLE.LocalSupplier.TestingIndicator] = false;
				}

				return (bool)this._drFacility[TABLE.LocalSupplier.TestingIndicator];
			}
			set
			{
				this._drFacility[TABLE.LocalSupplier.TestingIndicator] = value;

				//BR_6.09 - Facility must be Collection or Testing
				if ((value == false) && (this.IsCollectionFacility == false))
				{
					RuleBroken("FacilityIndicator", true);
					return;
//					string errorMessage =  Common.StrRes.SysErrMsg.Common.FacilityTypeIndicatorRequired().ResString;
//					throw new BusinessObjectException(errorMessage);
				}

				RuleBroken("FacilityIndicator", false);
			}
		}

		///<Developers>
		///<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2649"> 
		///		<ExpectedInput>guid</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="2653"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// If there are any active bloodproducts associated with this local supplier, return true
		/// Implemented BR_8.11
		/// </summary>
		public static bool IsLocalSupplierOnActiveBloodProducts(System.Guid localSupplierGuid)
		{
			//Per SMEs leave it as is for now and revisit later since the rule still may change
			//BR_8.11
			return DAL.LocalSupplier.IsLocalSupplierOnActiveBloodProducts(localSupplierGuid);
		}


		///<Developers>
		///<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2659"> 
		///		<ExpectedInput>fdaregnum</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="2660"> 
		///		<ExpectedInput>Invalid fdaregnum</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// If there are any active bloodunits associated with this local supplier, return true
		/// </summary>
		public static bool IsLocalSupplierOnActiveBloodUnits(string fdaRegistrationNumber)
		{

			return DAL.LocalSupplier.IsLocalSupplierOnActiveBloodUnits(fdaRegistrationNumber);
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6630"> 
		///		<ExpectedInput>localSupplierIccbbaNumber</ExpectedInput>
		///		<ExpectedOutput>Boolean</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6631"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BR_1.64
		/// </summary>
		public static bool IsLocalSupplierActiveByIccbbaRegistrationNumber(string localSupplierIccbbaNumber)
		{

			return DAL.LocalSupplier.IsLocalSupplierActiveByIccbbaRegistrationNumber(localSupplierIccbbaNumber);
		}


		///<Developers>
		///<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2661"> 
		///		<ExpectedInput>Common.RecordStatusCode</ExpectedInput>
		///		<ExpectedOutput>Common.RecordStatusCode</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="2663"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// RecordStatusCode - indicator of whether the local supplier is "Active" or "Inactive"
		/// </summary>
		public Common.RecordStatusCode RecordStatusCode
		{
			get
			{
				if (this._drFacility.IsNull(TABLE.LocalSupplier.RecordStatusCode))
				{
					this._drFacility[TABLE.LocalSupplier.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Unknown).ToString();
				}

				return Common.Utility.GetRecordStatusCodeFromString(_drFacility[TABLE.LocalSupplier.RecordStatusCode].ToString());
			}
			set
			{
				this._drFacility[TABLE.LocalSupplier.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(value);
			}
		}

	#endregion
	}
}
